<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * lib.php - Contains Plagiarism plugin specific functions called by Modules.
 *
 * @since 2.0
 * @package    plagiarism_similarity
 * @subpackage plagiarism
 * @author     Dan Marsden <dan@danmarsden.com>
 * @copyright  2011 Dan Marsden http://danmarsden.com
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();

// Get global class.
global $CFG;
require_once($CFG->dirroot.'/plagiarism/lib.php');
require_once($CFG->dirroot . '/lib/filelib.php');

/**
 * Class plagiarism_plugin_similarity
 *
 * @package   plagiarism_similarity
 * @copyright 2011 Dan Marsden
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

class plagiarism_plugin_similarity extends plagiarism_plugin {
    /**
     * This function should be used to initialise settings and check if plagiarism is enabled.
     *
     * @return mixed - false if not enabled, or returns an array of relevant settings.
     */
    public static function get_settings() {
        static $plagiarismsettings;
        if (!empty($plagiarismsettings) || $plagiarismsettings === false) {
            return $plagiarismsettings;
        }
        $plagiarismsettings = (array)get_config('plagiarism_similarity');
        // Check if enabled.
        if (isset($plagiarismsettings['enabled']) && $plagiarismsettings['enabled']) {
            // Now check to make sure required settings are set!
            if (empty($plagiarismsettings['api'])) {
                debugging("similarity API URL not set!");
                return false;
            }
            return $plagiarismsettings;
        } else {
            return false;
        }
    }

    /**
     * Generic handler function for all events - queues files for sending.
     * @param stdClass $eventdata
     * @return boolean
     */
    public function event_handler($eventdata) {
        global $DB, $CFG;

        $plagiarismsettings = $this->get_settings();
        if (!$plagiarismsettings) {
            return true;
        }
        $cmid = $eventdata['contextinstanceid'];

        // Check if the module associated with this event still exists.
        if (!$DB->record_exists('course_modules', array('id' => $cmid))) {
            return true;
        }

        $userid = $eventdata['userid'];
        $relateduserid = null;

        // Check if this is a submission on-behalf.
        if (!empty($eventdata['relateduserid'])) {
            $relateduserid = $eventdata['relateduserid'];
        }

        // Check to see if restrictcontent is in use.
        $showcontent = true;
        $showfiles = true;
        if (!empty($plagiarismvalues['similarity_restrictcontent'])) {
            if ($plagiarismvalues['similarity_restrictcontent'] == PLAGIARISM_similarity_RESTRICTCONTENTFILES) {
                $showcontent = false;
            } else if ($plagiarismvalues['similarity_restrictcontent'] == PLAGIARISM_similarity_RESTRICTCONTENTTEXT) {
                $showfiles = false;
            }
        }
        $charcount = plagiarism_similarity_charcount();

        // Normal situation: 1 or more assessable files attached to event, ready to be checked.
        if (!empty($eventdata['other']['pathnamehashes']) && $showfiles) {
            foreach ($eventdata['other']['pathnamehashes'] as $hash) {
                $fs = get_file_storage();
                $efile = $fs->get_file_by_hash($hash);

                if (empty($efile)) {
                    continue;
                } else if ($efile->get_filename() === '.') {
                    // This 'file' is actually a directory - nothing to submit.
                    continue;
                }
                $filehandle = $efile->get_content_file_handle();
                $streammeta = stream_get_meta_data($filehandle);

                // Get the physical file path.
                $physicalpath = $streammeta['uri'];
                $tempfilepath = $CFG->dirroot.'/plagiarism/similarity/files/'.$efile->get_filename();
                $parmanentfilepath = $CFG->wwwroot.'/plagiarism/similarity/files/'.$efile->get_filename();
                $tempfile = fopen($tempfilepath, "w") || die("Error: Unable to open file.");
                $filecontents = file_get_contents($physicalpath);
                $tempfile = file_put_contents($tempfilepath, $filecontents);
                $api = get_config('plagiarism_similarity', 'api');
                if($tempfile) {
                    $c = new curl(array('proxy' => true));
                    $c->setopt(array('CURLOPT_RETURNTRANSFER' => 1,
                        'CURLOPT_TIMEOUT' => 60, // Set to 60seconds just in case.
                        'CURLOPT_HTTPAUTH' => CURLAUTH_BASIC
                    ));
                    $filetype = mime_content_type($physicalpath);
                    $header = array();
                    $header[] = "x-doco-apikey: $api";
                    $header[] = 'X-Name: '.$efile->get_filename();
                    $header[] = "Content-Type: $filetype";
                    $header[] = "X-URL: $parmanentfilepath";
                    $header[] = "X-E-Mail: $CFG->noreplyaddress";
                    $header[] = "X-Lang: en";
                    $header[] = "X-PrivateStorage: 1";
                    $c->setHeader($header);
                    $baseurl = 'https://www.docoloc.de/rv1';
                    $html = $c->post($baseurl .'/jobs');
                    $response = $c->getResponse();
                }
            }
        }
        return $result;
    }

}

/**
 * Used to obtain allowed file types
 *
 * @param boolean $checkdb
 * @return array()
 */
function similarity_default_allowed_file_types($checkdb = false) {
    global $DB;
    $filetypes = array('doc'  => 'application/msword',
        'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'sxw'  => 'application/vnd.sun.xml.writer',
        'pdf'  => 'application/pdf',
        'txt'  => 'text/plain',
        'rtf'  => 'application/rtf',
        'html' => 'text/html',
        'htm'  => 'text/html',
        'wps'  => 'application/vnd.ms-works',
        'odt'  => 'application/vnd.oasis.opendocument.text',
        'pages' => 'application/x-iwork-pages-sffpages',
        'xls' => 'application/vnd.ms-excel',
        'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'ps' => 'application/postscript',
        'hwp' => 'application/x-hwp');

    if ($checkdb) {
        // Get all filetypes from db as well.
        $sql = 'SELECT name, value FROM {config_plugins} WHERE plugin = :plugin AND ' . $DB->sql_like('name', ':name');
        $types = $DB->get_records_sql($sql, array('name' => 'ext_%', 'plugin' => 'plagiarism_similarity'));
        foreach ($types as $type) {
            $ext = strtolower(str_replace('ext_', '', $type->name));
            $filetypes[$ext] = $type->value;
        }
    }

    return $filetypes;

}


/**
 * Function to list plugins that similarity supports.
 * @return array
 *
 */
function similarity_supported_modules() {
    global $CFG;
    $supportedmodules = array('assign');
    if (file_exists($CFG->dirroot.'/mod/hsuforum/version.php')) {
        $supportedmodules[] = 'hsuforum';
    }
    return $supportedmodules;
}

/**
 * Function to list question types that similarity supports.
 * @return array
 *
 */
function plagiarism_similarity_supported_qtypes() {
    return array('essay');
}


/**
 * Helper function to warn admin if Cron not running correctly.
 *
 * @throws coding_exception
 * @throws dml_exception
 *
 */
function plagiarism_similarity_checkcronhealth() {
    global $DB;
    $taskscores = $DB->get_record('task_scheduled', array('component' => 'plagiarism_similarity',
        'classname' => '\plagiarism_similarity\task\get_scores'));
    if (empty($taskscores) || $taskscores->lastruntime < time() - 3600 * 0.5) { // Check if run in last 30min.
        \core\notification::add(get_string('cronwarningscores', 'plagiarism_similarity'), \core\notification::ERROR);
    }

    $taskscores = $DB->get_record('task_scheduled', array('component' => 'plagiarism_similarity',
        'classname' => '\plagiarism_similarity\task\send_files'));
    if (empty($taskscores) || $taskscores->lastruntime < time() - 3600 * 0.5) { // Check if run in last 30min.
        \core\notification::add(get_string('cronwarningsendfiles', 'plagiarism_similarity'), \core\notification::ERROR);
    }
}


