<?php
/**
 * PHPCompatibility, an external standard for PHP_CodeSniffer.
 *
 * @package   PHPCompatibility
 * @copyright 2012-2022 PHPCompatibility Contributors
 * @license   https://opensource.org/licenses/LGPL-3.0 LGPL3
 * @link      https://github.com/PHPCompatibility/PHPCompatibility
 */

namespace PHPCompatibility\Sniffs\ParameterValues;

use PHPCompatibility\AbstractFunctionCallParameterSniff;
use PHP_CodeSniffer\Files\File;

/**
 * Detect: Calling mb_check_encoding() without any arguments is deprecated as of PHP 8.1.
 * This will become a fatal error as of PHP 9.0.
 *
 * PHP version 8.1
 * PHP version 9.0
 *
 * @link https://wiki.php.net/rfc/deprecations_php_8_1#mb_check_encoding_without_argument
 * @link https://www.php.net/manual/en/migration81.deprecated.php#migration81.deprecated.mbstring
 * @link https://www.php.net/manual/en/function.mb-check-encoding.php
 *
 * @since 10.0.0
 */
class RemovedMbCheckEncodingNoArgsSniff extends AbstractFunctionCallParameterSniff
{

    /**
     * Functions to check for.
     *
     * @since 10.0.0
     *
     * @var array
     */
    protected $targetFunctions = [
        'mb_check_encoding' => true,
    ];

    /**
     * Do a version check to determine if this sniff needs to run at all.
     *
     * @since 10.0.0
     *
     * @return bool
     */
    protected function bowOutEarly()
    {
        return ($this->supportsAbove('8.1') === false);
    }

    /**
     * Process the parameters of a matched function.
     *
     * @since 10.0.0
     *
     * @param \PHP_CodeSniffer\Files\File $phpcsFile    The file being scanned.
     * @param int                         $stackPtr     The position of the current token in the stack.
     * @param string                      $functionName The token content (function name) which was matched.
     * @param array                       $parameters   Array with information about the parameters.
     *
     * @return void
     */
    public function processParameters(File $phpcsFile, $stackPtr, $functionName, $parameters)
    {
        // Nothing to do. Function called with one or more parameters.
    }

    /**
     * Process the function if no parameters were found.
     *
     * @since 10.0.0
     *
     * @param \PHP_CodeSniffer\Files\File $phpcsFile    The file being scanned.
     * @param int                         $stackPtr     The position of the current token in the stack.
     * @param string                      $functionName The token content (function name) which was matched.
     *
     * @return void
     */
    public function processNoParameters(File $phpcsFile, $stackPtr, $functionName)
    {
        $phpcsFile->addWarning(
            'Calling %s() without arguments is deprecated since PHP 8.1.',
            $stackPtr,
            'Found',
            [$functionName]
        );
    }
}
